/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <iostream>
#include <cmath>
#include <sstream>
#include <algorithm>

#include "CalculatorTest.h"
#include "Calibration.h"

using namespace std;
using namespace LayerManagerCalibration;

CalculatorTest::CalculatorTest(string& fileName) : testFileName(fileName)
{
}

CalculatorTest::~CalculatorTest()
{
}

bool CalculatorTest::builtInTest()
{
    // Texas Instrument Example
    // taken from "Calibration in touch-screen systems"
    // slyt277.pdf - Example 1: Three-point calibration
    // http://www.ti.com/lit/an/slyt277/slyt277.pdf

    cout << "*** Built In Test 1 - TI Example ***" << endl;

    coordinate expected1;
    coordinate expected2;
    coordinate expected3;
    coordinate measured1;
    coordinate measured2;
    coordinate measured3;

    double coefficientExpectedXOne = 0.0623;
    double coefficientExpectedXTwo = 0.0054;
    double coefficientExpectedXThree = 9.9951;
    double coefficientExpectedYOne = -0.0163;
    double coefficientExpectedYTwo = 0.1868;
    double coefficientExpectedYThree = -10.1458;
    bool result = false;

    measured1.x = 678;
    measured1.y = 2169;
    measured2.x = 2807;
    measured2.y = 1327;
    measured3.x = 2629;
    measured3.y = 3367;

    expected1.x = 64;
    expected1.y = 384;
    expected2.x = 192;
    expected2.y = 192;
    expected3.x = 192;
    expected3.y = 576;

    result = checkCalibrationParameters(measured1,
                                        measured2,
                                        measured3,
                                        coefficientExpectedXOne,
                                        coefficientExpectedXTwo,
                                        coefficientExpectedXThree,
                                        coefficientExpectedYOne,
                                        coefficientExpectedYTwo,
                                        coefficientExpectedYThree,
                                        expected1,
                                        expected2,
                                        expected3);

    string testName = "Built In Test - TI, Example 1";

    displayTestResult(testName, result);

    return result;
}

bool CalculatorTest::processTestFile()
{
    bool result = true;
    string testName = "Tests from File";

    try
    {
         fileStream = new std::ifstream(testFileName.c_str());
    }
    catch (ifstream::failure& e) {
        displayTestResult(testName, false);
        throw e;
    }

    if (!fileStream->is_open())
    {
        cerr << "File Can't be opened" << endl;
        result = false;
    }
    else
    {
        string input;
        string token;
        vector<string> entries;
        int counter = 0;
        char divCharacter;

        while (getline (*fileStream, input))
        {
            istringstream line(input);

            entries.clear();
            counter++;

            if (count(input.begin(), input.end(), ',')
                      == (NO_ENTRIES_IN_TEST_CASE - 1))
            {
                divCharacter = ',';
            }
            else if (count(input.begin(), input.end(), ' ')
                           == (NO_ENTRIES_IN_TEST_CASE - 1))
            {
                divCharacter = ' ';
            }
            else
            {
                cerr << "Test Case not formed properly, "
                     << "number of entries incorrect for Test Entry: "
                     << counter << endl;
                result = false;
                continue;
            }

            while(getline(line, token, divCharacter))
            {
                entries.push_back(token);
            }

            calibrationTestCase testCase;

            if (processFileEntry(entries, testCase))
            {
                result = processTestCase(testCase) && result;
            }
            else
            {
                result = false;
                cerr << "Test Case not formed properly, "
                     << "for Test Entry: " << counter << endl;
            }
        }
    }

    displayTestResult(testName, result);

    return result;
}

void CalculatorTest::printCoordinates(const string& title,
                                      const coordinate& point1,
                                      const coordinate& point2,
                                      const coordinate& point3)
{
    cout << title << endl;
    cout << "(X = " << point1.x << ", Y = " << point1.y << " )" << endl;
    cout << "(X = " << point2.x << ", Y = " << point2.y << " )" << endl;
    cout << "(X = " << point3.x << ", Y = " << point3.y << " )" << endl;
    cout << endl;
}

void CalculatorTest::printCoefficients(const string& title,
                                       const double& coefficientExpectedXOne,
                                       const double& coefficientExpectedXTwo,
                                       const double& coefficientExpectedXThree,
                                       const double& coefficientExpectedYOne,
                                       const double& coefficientExpectedYTwo,
                                       const double& coefficientExpectedYThree,
                                       const double& coefficientCalculatedXOne,
                                       const double& coefficientCalculatedXTwo,
                                       const double& coefficientCalculatedXThree,
                                       const double& coefficientCalculatedYOne,
                                       const double& coefficientCalculatedYTwo,
                                       const double& coefficientCalculatedYThree)
{
    cout << title << endl << endl;

    cout << "Coefficient Expected X One " << coefficientExpectedXOne
         << " Coefficient Calculated X One " << coefficientCalculatedXOne
         << ", Difference is: "
         << coefficientCalculatedXOne - coefficientExpectedXOne
         << endl;

    cout << "Coefficient Expected X Two " << coefficientExpectedXTwo
         << " Coefficient Calculated X Two " << coefficientCalculatedXTwo
         << ", Difference is: "
         << coefficientCalculatedXTwo - coefficientExpectedXTwo
         << endl;

    cout << "Coefficient Expected X Three " << coefficientExpectedXThree
         << " Coefficient Expected X Three " << coefficientCalculatedXThree
         << ", Difference is: "
         << coefficientCalculatedXThree - coefficientExpectedXThree
         << endl;

    cout << "Coefficient Expected Y One " << coefficientExpectedYOne
         << " Coefficient Calculated Y One " << coefficientCalculatedYOne
         << ", Difference is: "
         << coefficientCalculatedYOne - coefficientExpectedYOne
         << endl;

    cout << "Coefficient Expected Y Two " << coefficientExpectedYTwo
         << " Coefficient Calculated Y Two " << coefficientCalculatedYTwo
         << ", Difference is: "
         << coefficientCalculatedYTwo - coefficientExpectedYTwo
         << endl;

    cout << "Coefficient Expected Y Three " << coefficientExpectedYThree
         << " Coefficient Expected Y Three " << coefficientCalculatedYThree
         << ", Difference is: "
         << coefficientCalculatedYThree - coefficientExpectedYThree
         << endl;

    cout << endl;
}

void CalculatorTest::printCoefficients(const std::string& title,
                                       const double& coefficientXOne,
                                       const double& coefficientXTwo,
                                       const double& coefficientXThree,
                                       const double& coefficientYOne,
                                       const double& coefficientYTwo,
                                       const double& coefficientYThree)
{
    cout << title << endl << endl;

    cout << "Coefficient X One " << coefficientXOne << endl;

    cout << "Coefficient X Two " << coefficientXTwo << endl;

    cout << "Coefficient X Three " << coefficientXThree << endl;

    cout << "Coefficient Y One " << coefficientYOne << endl;

    cout << "Coefficient Y Two " << coefficientYTwo << endl;

    cout << "Coefficient Y Three " << coefficientYThree << endl;

    cout << endl;
}

bool CalculatorTest::checkCalibrationParameters(const coordinate& measured1,
                                                const coordinate& measured2,
                                                const coordinate& measured3,
                                                const double& coefficientXOne,
                                                const double& coefficientXTwo,
                                                const double& coefficientXThree,
                                                const double& coefficientYOne,
                                                const double& coefficientYTwo,
                                                const double& coefficientYThree,
                                                const coordinate& expected1,
                                                const coordinate& expected2,
                                                const coordinate& expected3)
{

    float cXOne;
    float cXTwo;
    float cXThree;
    float cYOne;
    float cYTwo;
    float cYThree;
    string title;
    bool result = false;

    // Calculate the coefficients from the measured and
    // expected coordinates
    result = Calibration::calculateCoefficients(measured1,
                                                measured2,
                                                measured3,
                                                expected1,
                                                expected2,
                                                expected3,
                                                cXOne,
                                                cXTwo,
                                                cXThree,
                                                cYOne,
                                                cYTwo,
                                                cYThree);

    if (result)
    {
        // Print the Measured Coordinates
        title = "Measured Coordinates";
        printCoordinates(title, measured1, measured2, measured3);

        // Print the Expected Coordinates
        title = "Expected Coordinates";
        printCoordinates(title, expected1, expected2, expected3);

        // Print the Calculated, Expected Coefficients and
        // differences between the two
        title = "Coefficients:";
        printCoefficients(title,
                          cXOne,
                          cXTwo,
                          cXThree,
                          cYOne,
                          cYTwo,
                          cYThree,
                          (double)coefficientXOne,
                          (double)coefficientXTwo,
                          (double)coefficientXThree,
                          (double)coefficientYOne,
                          (double)coefficientYTwo,
                          (double)coefficientYThree);

        // Use the calculated coefficients to translate
        // the original given measured coordinate values,
        // compare the resultant coordinates against the
        // original expected coordinates. The translated
        // coordinates should equal the original expected.
        coordinate translated1;
        coordinate translated2;
        coordinate translated3;
        determineTranslatedCoordinates(measured1, measured2, measured3,
                                       translated1, translated2, translated3,
                                       coefficientXOne, coefficientXTwo,
                                       coefficientXThree, coefficientYOne,
                                       coefficientYTwo, coefficientYThree);

        // Print the translated coordinates
        title = "Coordinates Translated by Calculated Coefficients";
        printCoordinates(title, translated1, translated2, translated3);

        // Compare the expected coordinated against the translated
        result = compareCoordinateSets(expected1, expected2, expected3,
                                       translated1, translated2, translated3);
    }

    return result;
}

bool CalculatorTest::deriveCalibrationCoefficients(const coordinate& measured1,
                                                   const coordinate& measured2,
                                                   const coordinate& measured3,
                                                   double& coefficientXOne,
                                                   double& coefficientXTwo,
                                                   double& coefficientXThree,
                                                   double& coefficientYOne,
                                                   double& coefficientYTwo,
                                                   double& coefficientYThree,
                                                   const coordinate& expected1,
                                                   const coordinate& expected2,
                                                   const coordinate& expected3)
{
    float xCoefficientOne;
    float xCoefficientTwo;
    float xCoefficientThree;
    float yCoefficientOne;
    float yCoefficientTwo;
    float yCoefficientThree;

    bool result = false;
    string title;

    // Calculate the coefficients from the measured and
    // expected coordinates
    result = Calibration::calculateCoefficients(measured1,
                                                measured2,
                                                measured3,
                                                expected1,
                                                expected2,
                                                expected3,
                                                xCoefficientOne,
                                                xCoefficientTwo,
                                                xCoefficientThree,
                                                yCoefficientOne,
                                                yCoefficientTwo,
                                                yCoefficientThree);

    if (result)
    {
        // Set the values
        coefficientXOne   = xCoefficientOne;
        coefficientXTwo   = xCoefficientTwo;
        coefficientXThree = xCoefficientThree;
        coefficientYOne   = yCoefficientOne;
        coefficientYTwo   = yCoefficientTwo;
        coefficientYThree = yCoefficientThree;

        // Print the Measured Coordinates
        title = "Measured Coordinates";
        printCoordinates(title, measured1, measured2, measured3);

        // Print the Expected Coordinates
        title = "Expected Coordinates";
        printCoordinates(title, expected1, expected2, expected3);

        // Print the CalculatedCoefficients
        title = "Calculated Coefficients:";
        printCoefficients(title,
                          xCoefficientOne,
                          xCoefficientTwo,
                          xCoefficientThree,
                          yCoefficientOne,
                          yCoefficientTwo,
                          yCoefficientThree);

        // Use the calculated coefficients to translate
        // the original given measured coordinate values,
        // compare the resultant coordinates against the
        // original expected coordinates. The translated
        // coordinates should equal the original expected.
        coordinate translated1;
        coordinate translated2;
        coordinate translated3;
        determineTranslatedCoordinates(measured1, measured2, measured3,
                                       translated1, translated2, translated3,
                                       xCoefficientOne, xCoefficientTwo,
                                       xCoefficientThree, yCoefficientOne,
                                       yCoefficientTwo, yCoefficientThree);

        // Print the translated coordinates
        title = "Coordinates Translated by Calculated Coefficients";
        printCoordinates(title, translated1, translated2, translated3);

        // Compare the expected coordinated against the translated
        result = compareCoordinateSets(expected1, expected2, expected3,
                                       translated1, translated2, translated3);
    }
    else
    {
        cerr << "Coefficient calculation can't be completed" << endl;
        result = false;
    }

    return result;
}

uint CalculatorTest::calibratedInterpolatedXCoord(const coordinate& point, 
                                                  const double& coefficientXOne,
                                                  const double& coefficientXTwo,
                                                  const double& coefficientXThree)
{
    double result = (coefficientXOne * point.x)
                    + (coefficientXTwo * point.y) + coefficientXThree;

    return round(result);
}

uint CalculatorTest::calibratedInterpolatedYCoord(const coordinate& point, 
                                                  const double& coefficientYOne,
                                                  const double& coefficientYTwo,
                                                  const double& coefficientYThree)
{
    double result = (coefficientYOne * point.x)
                    + (coefficientYTwo * point.y) + coefficientYThree;

    return round(result);
}

void CalculatorTest::determineTranslatedCoordinates(const coordinate& measured1,
                                                    const coordinate& measured2,
                                                    const coordinate& measured3,
                                                    coordinate& translated1,
                                                    coordinate& translated2,
                                                    coordinate& translated3,
                                                    const double& coefficientXOne,
                                                    const double& coefficientXTwo,
                                                    const double& coefficientXThree,
                                                    const double& coefficientYOne,
                                                    const double& coefficientYTwo,
                                                    const double& coefficientYThree)
{
    translated1.x = calibratedInterpolatedXCoord(measured1,
                                                 coefficientXOne,
                                                 coefficientXTwo,
                                                 coefficientXThree);
    translated2.x = calibratedInterpolatedXCoord(measured2,
                                                 coefficientXOne,
                                                 coefficientXTwo,
                                                 coefficientXThree);
    translated3.x = calibratedInterpolatedXCoord(measured3,
                                                 coefficientXOne,
                                                 coefficientXTwo,
                                                 coefficientXThree);

    translated1.y = calibratedInterpolatedYCoord(measured1,
                                                 coefficientYOne,
                                                 coefficientYTwo,
                                                 coefficientYThree);
    translated2.y = calibratedInterpolatedYCoord(measured2,
                                                 coefficientYOne,
                                                 coefficientYTwo,
                                                 coefficientYThree);
    translated3.y = calibratedInterpolatedYCoord(measured3,
                                                 coefficientYOne,
                                                 coefficientYTwo,
                                                 coefficientYThree);
}

bool CalculatorTest::compareCoordinateSets(const coordinate& expected1,
                                           const coordinate& expected2,
                                           const coordinate& expected3,
                                           const coordinate& translated1,
                                           const coordinate& translated2,
                                           const coordinate& translated3)
{
    return (    (expected1.x == translated1.x)
             && (expected1.y == translated1.y)
             && (expected2.x == translated2.x)
             && (expected2.y == translated2.y)
             && (expected3.x == translated3.x)
             && (expected3.y == translated3.y));
}

bool CalculatorTest::processTestCase(calibrationTestCase& testCase)
{
    bool testResult = false;
    bool caseResult = false;

    cout << endl << "*** " << testCase.strTestName << " ***" << endl;

    if (testCase.deriveCoefficient)
    {
        caseResult =
            deriveCalibrationCoefficients(testCase.measured1,
                                          testCase.measured2,
                                          testCase.measured3,
                                          testCase.coefficientXOne,
                                          testCase.coefficientXTwo,
                                          testCase.coefficientXThree,
                                          testCase.coefficientYOne,
                                          testCase.coefficientYTwo,
                                          testCase.coefficientYThree,
                                          testCase.expected1,
                                          testCase.expected2,
                                          testCase.expected3);
    }
    else
    {
        caseResult =
            checkCalibrationParameters(testCase.measured1,
                                       testCase.measured2,
                                       testCase.measured3,
                                       testCase.coefficientXOne,
                                       testCase.coefficientXTwo,
                                       testCase.coefficientXThree,
                                       testCase.coefficientYOne,
                                       testCase.coefficientYTwo,
                                       testCase.coefficientYThree,
                                       testCase.expected1,
                                       testCase.expected2,
                                       testCase.expected3);
    }

    testResult = (caseResult == testCase.expectedResult);

    displayTestResult(testCase.strTestName, testResult);

    return testResult;
}

bool CalculatorTest::processFileEntry(const vector<string>& entrys,
                                      calibrationTestCase& testCase)
{
    bool result = true;

    testCase.strTestName = entrys[0];
    result &= castStringToInt(entrys[1], testCase.measured1.x);
    result &= castStringToInt(entrys[2], testCase.measured1.y);
    result &= castStringToInt(entrys[3], testCase.measured2.x);
    result &= castStringToInt(entrys[4], testCase.measured2.y);
    result &= castStringToInt(entrys[5], testCase.measured3.x);
    result &= castStringToInt(entrys[6], testCase.measured3.y);
    result &= castStringToInt(entrys[7], testCase.expected1.x);
    result &= castStringToInt(entrys[8], testCase.expected1.y);
    result &= castStringToInt(entrys[9], testCase.expected2.x);
    result &= castStringToInt(entrys[10], testCase.expected2.y);
    result &= castStringToInt(entrys[11], testCase.expected3.x);
    result &= castStringToInt(entrys[12], testCase.expected3.y);

    if (   (entrys[13].compare("?") == 0)
        && (entrys[14].compare("?") == 0)
        && (entrys[15].compare("?") == 0)
        && (entrys[16].compare("?") == 0)
        && (entrys[17].compare("?") == 0)
        && (entrys[18].compare("?") == 0))
    {
        testCase.deriveCoefficient = true;
    }
    else
    {
        testCase.deriveCoefficient = false;

        result &= castStringToDouble(entrys[13], testCase.coefficientXOne);
        result &= castStringToDouble(entrys[14], testCase.coefficientXTwo);
        result &= castStringToDouble(entrys[15], testCase.coefficientXThree);
        result &= castStringToDouble(entrys[16], testCase.coefficientYOne);
        result &= castStringToDouble(entrys[17], testCase.coefficientYTwo);
        result &= castStringToDouble(entrys[18], testCase.coefficientYThree);
    }

    if (!entrys[19].compare("PASS"))
    {
        testCase.expectedResult = true;
    }
    else if (!entrys[19].compare("FAIL"))
    {
        testCase.expectedResult = false;
    }
    else
    {
        result = false;
    }

    return result;
}

bool CalculatorTest::castStringToInt(const string& entry, uint& value)
{
    istringstream iss(entry);

    iss >> value;

    if (iss.fail())
    {
        cerr << "CalibrationTest: " <<
                  "Found invalid value: " << entry;
        return false;
    }
    else
    {
        return true;
    }
}

bool CalculatorTest::castStringToDouble(const string& entry, double& value)
{
    istringstream iss(entry);

    iss >> value;

    if (iss.fail())
    {
        cerr << "CalibrationTest: " <<
                  "Found invalid value: " << entry;
        return false;
    }
    else
    {
        return true;
    }
}

void CalculatorTest::displayTestResult(string& testPoint, bool result)
{
    //Test equivalence
    if (!result)
    {
        cerr << "TEST FAILURE - " << testPoint << endl;
        cerr << endl;
    }
    else
    {
        cout << "TEST PASS - " << testPoint << endl;
        cout << endl;
    }
}
